/*=================================================================
 *
 * remove_artefacts_c.c ... Remove numerical artefacts from FFT:
 *                          -) force unimodaility
 *                          -) normalize density
 *
 * The calling syntax is:
 *		remove_artefacts_c(density, dz)
 *
 *  density     ... vector with density at dz*i, i=0,1,2,...
 *  dz          ... spacing of density points
 *
 * Sample call:
 *      remove_artefacts_c([1 0.9 1.5 0.5 0 1.2], 0.5)
 *
 *=================================================================*/

#include <math.h>
#include "mex.h"

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/* Recursive computation of loss distribution */
static void remove_artefacts_c(double density[], double dz[], unsigned int num_points)
{
    int i, max_pos = 0;
    double sum, scaling, max_value = 0;
    
    /* Determine position of maximum and remove negative values */
    for (i=0; i<num_points; i++)
    {
        density[i] = max(0, density[i]);
        if (density[i] > max_value) {
            max_pos = i;
            max_value = density[i];
        }
    }
    
    /* Force unimodality left of maximum */
    for (i=max(0,max_pos-1); i>=0; i--)
    {
        density[i] = -max(-density[i], -density[i+1]);
    }
    
    /* Force unimodality right of maximum */
    for (i=max_pos+1; i<num_points; i++)
    {
        density[i] = -max(-density[i], -density[i-1]);
    }
    
    /* Normalize density */
    sum = 0;
    for (i=0; i<num_points; i++)
    {
        sum += density[i];
    }
    scaling = 1.0 / (sum*dz[0]);
    for (i=0; i<num_points; i++)
    {
        density[i] *= scaling;
    }
    return;
}


/* Gateway routine (to Matlab) */
void mexFunction( int nlhs, mxArray *plhs[], 
		  int nrhs, const mxArray*prhs[] )
     
{ 
    double *density, *dz; 
    unsigned int num_points1, num_points2, num_points; 

    /* Get number of data points */
    num_points1 = mxGetM(prhs[0]);
    num_points2 = mxGetN(prhs[0]);
    num_points = max(num_points1, num_points2);
    
    /* Assign pointers to the various parameters */ 
    density = mxGetPr(prhs[0]);
    dz = mxGetPr(prhs[1]);
    plhs[0] = prhs[0];
    
    /* Do the actual computations in a subroutine */
    remove_artefacts_c(density, dz, num_points);
    return;
}
